/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
* FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
* SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
* CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
* CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
****************************************************************************/

#include "WindowSystems/WaylandGal2dWindowSystem.h"
#include "Log.h"
#include <fcntl.h>
#include <iomanip>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>

WaylandGal2dWindowSystem::WaylandGal2dWindowSystem(ICommandExecutor& executor, const char* displayname, int width, int height, Scene* pScene, InputManager* pInputManager)
: WaylandBaseWindowSystem(executor, displayname, width, height, pScene, pInputManager)
,m_pIpManagerList()
,m_evdevInputEvent(NULL)
{
    // Because of using DrmMode, it's not necessary to use frame timer.
    m_bUseFrameTimer = false;
    if (NULL != pInputManager)
    {
        m_pIpManagerList.push_back(pInputManager);
    }
    LOG_DEBUG("WaylandGal2dWindowSystem",
              "Creating, width=" << width << ", height=" << height << ", "
              "display name=" << displayname);
}

WaylandGal2dWindowSystem::~WaylandGal2dWindowSystem()
{
    LOG_DEBUG("WaylandGal2dWindowSystem","destructor called");
}

bool WaylandGal2dWindowSystem::initGraphicSystem()
{
    bool ret = graphicSystem->init((void*)NULL, (void*)m_wlDisplay);
    if (true != ret)
    {
        LOG_WARNING("WaylandGal2dWindowSystem",
                    "Failed to init graphic system");
        return false;
    }
    else
    {
        LOG_DEBUG("WaylandGal2dWindowSystem",
                  " success");
    }

    graphicSystem->updateScreenList(m_pScene->getScreenList());

    return true;

}

void WaylandGal2dWindowSystem::checkForNewSurfaceNativeContent()
{
    m_pScene->lockScene();
    const std::map<unsigned int, Surface*> surfaces = m_pScene->getAllSurfaces();
    for (std::map<unsigned int, Surface*>::const_iterator currentS = surfaces.begin(); currentS != surfaces.end(); ++currentS)
    {
       Surface* currentSurface = (*currentS).second;
       if (currentSurface->hasNativeContent())
       {
           LOG_DEBUG("WaylandGal2dWindowSystem",
                     "Allocate Platform Surface, "
                     "layer ID=" << currentSurface->getContainingLayerId() << ", "
                     "surface ID=" << currentSurface->getID());
           allocatePlatformSurface(currentSurface);
       }
       else // While we are at it, also cleanup any stale native content
       {
           LOG_DEBUG("WaylandGal2dWindowSystem",
                     "De-allocate Platform Surface"
                     "layer ID=" << currentSurface->getContainingLayerId() << ", "
                     "surface ID=" << currentSurface->getID());
           deallocatePlatformSurface(currentSurface);
       }
     }
    m_pScene->unlockScene();
}

bool WaylandGal2dWindowSystem::createNativeContext()
{
    /*nothing to do for gal2d window system*/
    return true;
}

bool WaylandGal2dWindowSystem::isScreenWiseRenderingNeeded(void)
{
    return true;
}

void WaylandGal2dWindowSystem::initRepaintTimer(void)
{
    LmScreenList screenList = m_pScene->getScreenList();
    struct wl_event_loop *loop = wl_display_get_event_loop(m_wlDisplay);

    for (LmScreenListIterator iter = screenList.begin();
         iter != screenList.end();
         ++iter)
    {
        struct repaint_data* repaintData = new struct repaint_data();
        repaintData->handleRepaintTimer = wl_event_loop_add_timer(loop, repaintTimerHandler, repaintData);
        repaintData->screenId = (*iter)->getID();
        repaintData->baseWindowSystem = (BaseWindowSystem*)this;
        repaintData->bRepaintNeeded = 0;
        repaintData->bRepaintScheduled = 0;
        repaintData->bRepaintTimerTriggered = 0;
        repaintData->bScreenDamaged = 0;
        repaintData->repaintWindowPeriodMS = 0;
        repaintData->repaintCount = 0;
        repaintData->bUseRepaintTimer = false;
        repaintData->systemState = IDLE_STATE;
        repaintData->lastSubmittedSurface = NULL;
        wl_list_init(&repaintData->listFrameCallback);

        m_repaintData.insert(std::pair<uint, struct repaint_data*>((*iter)->getID(),
                             repaintData));
    }
}

void WaylandGal2dWindowSystem::RedrawAllLayers(bool clear, bool swap)
{
    unsigned int screenID = 0;
    LmScreenList screenList = m_pScene->getScreenList();
    LmScreenListIterator iter = screenList.begin();
    LmScreenListIterator iterEnd = screenList.end();
    std::map<uint, struct repaint_data*>::iterator match;
    struct repaint_data* repaintData;

    LOG_DEBUG("WaylandGal2dWindowSystem",
              "No screens=" << screenList.size() << ", "
              "clear=" << clear << ", swap=" << swap);

    std::list<int> screensRedrawn;

    for (; iter != iterEnd; ++iter)
    {
        screenID = (*iter)->getID();

        if (m_currentScreenID != screenID)
        {
            LOG_DEBUG("WaylandGal2dWindowSystem",
                      "screenID=" << screenID << " "
                      "is not current, so redraw not done");
            continue;
        }

        if ( graphicSystem->isFlipPending(screenID))
        {
            LOG_DEBUG("WaylandGal2dWindowSystem",
                        "Flip is pending no redraw, "
                        "screen ID=" << screenID << ", "
                        "clear=" << clear << ", swap=" << swap);
            continue;
        }

        match = m_repaintData.find(screenID);
        if (match != m_repaintData.end()) {
            repaintData = match->second;
        }
        else {
            LOG_ERROR("WaylandGal2dWindowSystem",
                "No repaint data found in m_repaintData for screenID=" << screenID);
            continue;
        }

        LayerList layers = m_pScene->getCurrentRenderOrder(screenID);
        LayerList swLayers;
        if (layers.size() == 0)
            continue;
        // TODO: bRedraw is overly conservative if layers includes a hardware layer
        bool bRedraw = m_forceComposition || repaintData->bScreenDamaged || (repaintData->systemState == REDRAW_STATE);

        LOG_DEBUG("WaylandGal2dWindowSystem",
                "screenID=" << screenID << ", " <<
                "m_forceComposition=" << m_forceComposition << ", " <<
                "needsRedraw=" << repaintData->bScreenDamaged << ", " <<
                "systemState=" << repaintData->systemState);

        /*check the synchronized surface separately to overrule all redraw reasons*/
        if (graphicSystem->synchronizedSurfacesDamaged(layers) == false)
        {
            bRedraw = false;
        }

        if (bRedraw)
        {
            graphicSystem->switchScreen(screenID);
            graphicSystem->activateGraphicContext();
            if (clear)
            {
                graphicSystem->clearBackground();
            }
        }
        for(std::list<Layer*>::const_iterator current = layers.begin(); current != layers.end(); current++)
        {
            if ((*current)->getLayerType() == Hardware)
            {
                if (m_forceComposition || graphicSystem->needsRedraw(*current))
                {
                    renderHWLayer(*current);
                }
            }
            else if (bRedraw)
            {
                swLayers.push_back(*current);
            }
        }
        if (bRedraw)
        {
            graphicSystem->renderSWLayers(swLayers, false); // Already cleared
            if (swap)
            {
                graphicSystem->swapBuffers();
            }
            graphicSystem->releaseGraphicContext();

            if (m_debugMode)
            {
                printDebug();
            }

            screensRedrawn.push_back(screenID);

            struct screen_data *screenData;
            std::map<uint, struct screen_data*>::iterator match =
                m_screenData.find(screenID);
            if (match == m_screenData.end())
            {
                screenData = new struct screen_data();
                m_screenData.insert(std::pair<uint, struct screen_data*>(screenID, screenData));
            }
            else
            {
                screenData = match->second;
            }
            screenData->renderCounter++;

            calculateFps(screenID);
        }

        if (repaintData) {
            struct native_frame_callback* cb;
            struct native_frame_callback* cnext;

            LOG_DEBUG("WaylandGal2dWindowSystem",
                      "frameCallback flush starts for screenID=" << screenID);

            repaintData->bRedrawDone = bRedraw;
            repaintData->bRepaintNeeded = false;

            /*dispatch frame callbacks after repaint*/
            wl_list_for_each_safe(cb, cnext, &repaintData->listFrameCallback, link) {

                LOG_DEBUG("WaylandGal2dWindowSystem",
                          "Callback=" << cb << " is sent");

                wl_callback_send_done(&cb->resource, getTime());
                wl_resource_destroy(&cb->resource);
            }

            if (bRedraw) {
                repaintData->bScreenDamaged = false;
                repaintData->systemState = IDLE_STATE;
            }
        }
    }

    gettimeofday(&tv0, NULL);

    // Clear damage for every screen that was redrawn
    for (std::list<int>::iterator cur = screensRedrawn.begin();
         cur != screensRedrawn.end();
         cur++)
    {
        ClearDamage(*cur);
    }
}

bool WaylandGal2dWindowSystem::createInputEvent()
{
    m_evdevInputEvent = new WaylandEvdevInputEvent(this);
    m_inputEvent = m_evdevInputEvent;
    m_inputEvent->setupInputEvent();
    LOG_DEBUG("WaylandGal2dWindowSystem",
              "Created input event=" << m_inputEvent);
    return true;
}


WlInputEventList* WaylandGal2dWindowSystem::getInputEventList()
{
    WlInputEventList* eventList = NULL;

    if (NULL != m_evdevInputEvent)
    {
        eventList = m_evdevInputEvent->getInputEventList();
    }
    return eventList;
}

void WaylandGal2dWindowSystem::manageWLInputEvent(const InputDevice type, InputEventState state,
                                                  const WLEvent *wlEvent, WaylandInputEvent *pWlIpEvent,
                                                  char *pSeatName, int displayID)
{
    LOG_INFO("WaylandGal2dWindowSystem",
              "Called, device type=" << type << ", "
              "state=" << state << ", ");

    if (!pWlIpEvent)
    {
        LOG_WARNING("WaylandGal2dWindowSystem", "InputEvent not available");
        return;
    }
    Surface *surface = NULL;
    native_surface *nativeSurface = NULL;
    uint32_t time = getTime();

     /* Locking scene mutex */
    m_pScene->lockScene();

    switch (type)
    {
    case INPUT_DEVICE_KEYBOARD:
        {
            LOG_DEBUG("WaylandGal2dWindowSystem",
                        "INPUT_DEVICE_KEYBOARD: state=" << state << ", "
                        "keyCode=" << wlEvent->keyCode << ", "
                        "mutex lock=" << &this->run_lock);
            surface = getInputManager(pSeatName)->reportKeyboardEvent(state, wlEvent->keyCode);
            if (!surface)
            {
                LOG_WARNING("WaylandGal2dWindowSystem",
                              "INPUT_DEVICE_KEYBOARD: key event with state=" << state << ", "
                              "keyCode=" << wlEvent->keyCode << " "
                              "is ignored as no surface has keyboard focus");
                pWlIpEvent->inputDevice().setKeyboardFocus(NULL);
                break;
            }

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandGal2dWindowSystem",
                            "KEYBOARD nativeSurface for surface, ID="<< surface->getID()
                            << " not found" << ", mutex lock=" << &this->run_lock);
                break;
            }
            LOG_INFO("WaylandGal2dWindowSystem",
                      "INPUT_DEVICE_KEYBOARD: keycode "<< wlEvent->keyCode <<
                      "state: "<< state <<" send"
                      "to ilm surface ID="<< surface->getID() << ", "
                      "native surface="<< nativeSurface << ", "
                      "mutex lock=" << &this->run_lock);
            switch (state)
            {
            case INPUT_STATE_PRESSED:
                  pWlIpEvent->inputDevice().sendKeyPressEvent(
                    &nativeSurface->surface, time, wlEvent->keyCode);
                break;
            case INPUT_STATE_RELEASED:
                  pWlIpEvent->inputDevice().sendKeyReleaseEvent(
                    &nativeSurface->surface, time, wlEvent->keyCode);
                break;
            case INPUT_STATE_OTHER:
            default:
                // nothing to do
                break;
            }
        }
        break;
    case INPUT_DEVICE_POINTER:
        {
            LOG_DEBUG("WaylandGal2dWindowSystem",
                      "INPUT_DEVICE_POINTER: state=" << state << ", "
                      "x=" << wlEvent->x << ", "
                      "y=" << wlEvent->y << ", "
                      "axis=" << wlEvent->axis << ", "
                      "value=" << wlEvent->axisValue << ", "
                      "mutex lock=" << &this->run_lock);
            Point globalPos = {state, wlEvent->x, wlEvent->y};
            Point localPos  = globalPos;
            surface = getInputManager(pSeatName)->reportPointerEvent(localPos, displayID);
            if (!surface){
                LOG_WARNING("WaylandGal2dWindowSystem",
                            "Surface Not Found!, "
                            "INPUT_DEVICE_POINTER: state=" << state << ", "
                            "axis=" << wlEvent->axis << ", "
                            "value=" << wlEvent->axisValue << ", "
                            "x=" << wlEvent->x << ", "
                            "y=" << wlEvent->y << ", "
                            "mutex lock=" << &this->run_lock);
                if (state == INPUT_STATE_PRESSED &&
                    pWlIpEvent->inputDevice().pointerDevice() != NULL)
                {
                    pWlIpEvent->inputDevice().pointerDevice()->button_count--;
                }

                break;
            }
            LOG_DEBUG("WaylandGal2dWindowSystem",
                      "INPUT_DEVICE_POINTER: Local coordinates: "
                      "x=" << localPos.x << ", "
                      "y=" << localPos.y << ", "
                      "axis="<<  wlEvent->axis << ", "
                      "mutex lock=" << &this->run_lock);

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandGal2dWindowSystem",
                            "INPUT_DEVICE_POINTER: "
                            "Native Surface not found for surface, "
                            "ID="<< surface->getID() << ", "
                            "mutex lock=" << &this->run_lock);

                if (state == INPUT_STATE_PRESSED &&
                    pWlIpEvent->inputDevice().pointerDevice() != NULL)
                {
                    pWlIpEvent->inputDevice().pointerDevice()->button_count--;
                }

                break;
            }
            LOG_INFO("WaylandGal2dWindowSystem",
                      "INPUT_DEVICE_POINTER: "
                      "send event to ilm surface ID="<< surface->getID() << ", "
                      "state " << state << ", "
                      "local coord x=" << localPos.x << ", y=" << localPos.y << ", "
                      "axis " <<  wlEvent->axis <<", "
                      "axisValue " <<  wlEvent->axisValue <<", "
                      "time "<< time << ", "
                      "mutex lock=" << &this->run_lock);

            switch (state){
            case INPUT_STATE_PRESSED:
                pWlIpEvent->inputDevice().sendMousePressEvent(
                    &nativeSurface->surface, globalPos, localPos, wlEvent->button, time);
                break;
            case INPUT_STATE_RELEASED:
                pWlIpEvent->inputDevice().sendMouseReleaseEvent(
                    globalPos, localPos, wlEvent->button, time);
                break;
            case INPUT_STATE_MOTION:
                pWlIpEvent->inputDevice().sendMouseMotionEvent(
                    &nativeSurface->surface, globalPos, localPos, time);
                break;
            case INPUT_STATE_AXIS:
                pWlIpEvent->inputDevice().sendAxisEvent(&nativeSurface->surface,
                                                         wlEvent->axis,
                                                         wlEvent->axisValue,
                                                         time);
                break;
            case INPUT_STATE_OTHER:
            default:
                break;
            }
        }
        break;
    case INPUT_DEVICE_TOUCH:
        {
            LOG_DEBUG("WaylandGal2dWindowSystem",
                        "INPUT_DEVICE_TOUCH: state=" << state << ", "
                        "x=" << wlEvent->x << ", "
                        "y=" << wlEvent->y << ", "
                        "mutex lock=" << &this->run_lock);
            if (wlEvent->touchId)
                state = INPUT_STATE_OTHER;

            Point pt = {state, wlEvent->x, wlEvent->y};
            PointVect ptVec(1, pt);
            surface = getInputManager(pSeatName)->reportTouchEvent(ptVec, displayID);

            if (!surface) {
                if (pWlIpEvent->inputDevice().touchDevice()->focus)
                {
                    pWlIpEvent->inputDevice().cleanupTouchFocus();
                }
                break;
            }

            nativeSurface = getNativeSurfaceFromSurface(surface);
            if (!nativeSurface)
            {
                LOG_WARNING("WaylandGal2dWindowSystem",
                            "INPUT_DEVICE_TOUCH: no native surface found "
                            "for surface, ID="<< surface->getID() << ", "
                            "mutex lock=" << &this->run_lock);
                break;
            }
            LOG_INFO("WaylandGal2dWindowSystem",
                      "INPUT_DEVICE_TOUCH: "
                      "send to ilm surface ID=" << surface->getID() << ", "
                      "state "<< state <<", "
                      "touch id "<< wlEvent->touchId <<", "
                      "local coord x=" << ptVec[0].x << ", y=" << ptVec[0].y << ", "
                      "time "<< time );
            switch (state)
            {
            case INPUT_STATE_PRESSED:
            case INPUT_STATE_MOTION:
            case INPUT_STATE_RELEASED:
            case INPUT_STATE_OTHER:
                pWlIpEvent->inputDevice().sendTouchPointEvent(
                    &nativeSurface->surface, time, wlEvent->touchId,
                    wlEvent->touchType, ptVec[0]);
                break;
            default:
                break;
            }
        }
        break;
    case INPUT_DEVICE_ALL:
    default:
        break;
    }

    /* Unlocking scene mutex */
    m_pScene->unlockScene();
}


InputManager* WaylandGal2dWindowSystem::addInputManager(char *seatName)
{
    IpManagerList::iterator it;
    bool seatExists = false;
    InputManager *pInputManager = NULL;

    for (it = m_pIpManagerList.begin();
        it != m_pIpManagerList.end() && (seatExists == false);it++)
    {
        if (!strcmp((*it)->getSeatName(), seatName))
        {
            seatExists = true;
            pInputManager = (*it);
        }
    }
    if (false == seatExists)
    {
        pInputManager = new InputManager(m_pScene, m_pInputManager->m_config);
        if (NULL == pInputManager)
        {
            LOG_ERROR("WaylandGal2dWindowSystem", "Failed to allocate input manager");
        }
        else
        {
            pInputManager->setSeatName(seatName);
            pInputManager->setGraphicSystem(m_pInputManager->m_pGraphicSystem);
            m_pIpManagerList.push_back(pInputManager);
        }
    }
    return pInputManager;
}

InputManager* WaylandGal2dWindowSystem::getInputManager(char *seatName)
{
    IpManagerList::iterator it;
    bool seatExists = false;
    InputManager *pIpManager = NULL;
    for (it = m_pIpManagerList.begin();
        it != m_pIpManagerList.end() && (seatExists == false);it++)
    {
        if (!strcmp((*it)->getSeatName(), seatName))
        {
            seatExists = true;
            pIpManager = *it;
        }
    }
    return pIpManager;
}

bool WaylandGal2dWindowSystem::getSupportedSeats(ilmInputDevice bitmask,
                       t_ilm_string ** pSeatArray,
                       unsigned int *pSizeSeatArray)
{
    bool ret = true;
    t_ilm_string *pStrArray = NULL;
    IpManagerList::iterator it;
    ilmInputDevice deviceBtmask = 0;
    unsigned int matchingSeats = 0;
    unsigned int strCount = 0;
    *pSizeSeatArray = 0;
    for (it = m_pIpManagerList.begin();
        it != m_pIpManagerList.end();it++)
    {
        (*it)->getCurrentDevices(&deviceBtmask);
        if ((bitmask & deviceBtmask) == bitmask)
        {
            matchingSeats++;
        }
        deviceBtmask = 0;
    }
    if (matchingSeats > 0)
    {
        pStrArray = (t_ilm_string *)malloc(matchingSeats * sizeof(t_ilm_string));
        if (NULL != pStrArray)
        {
            for (it = m_pIpManagerList.begin();
                it != m_pIpManagerList.end();it++)
            {
                (*it)->getCurrentDevices(&deviceBtmask);
                if ((bitmask & deviceBtmask) == bitmask)
                {
                    pStrArray[strCount] = strdup((const char*)(*it)->getSeatName());
                    if (NULL != pStrArray[strCount])
                    {
                        strCount++;
                        *pSizeSeatArray = strCount;
                    }
                    else
                    {
                        ret = false;
                        LOG_ERROR("WaylandGal2dWindowSystem", "Unable allocate memory for seat name");
                        break;
                    }
                }
                deviceBtmask = 0;
            }
        }
        else
        {
            ret = false;
            LOG_ERROR("WaylandGal2dWindowSystem", "Unable allocate memory for seat list");
        }
    }
    *pSeatArray = pStrArray;

    return ret;
}

